<?php

namespace Modules\data\Controllers;

use App\Controllers\BaseController;
use App\Libraries\DatatableQuery;
use App\Traits\CommonTrait;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use Modules\data\Models\EtatLieuModel;
use Modules\Params\Models\BrancheModel;
use Modules\Params\Models\CommuneModel;
use Psr\Log\LoggerInterface;
use stdClass;
use UConverter;

use function PHPUnit\Framework\fileExists;

class EtatLieuController extends BaseController
{
    public EtatLieuModel $etatLieuModel;
    public function initController(RequestInterface $request, ResponseInterface $response, LoggerInterface $logger)
    {
        parent::initController($request, $response, $logger);
        $this->etatLieuModel = new EtatLieuModel();
    }

    public function index()
    {
        $this->viewData['url_list'] = "/data/etat/data";
        $this->viewData['js'] = "Common.js";
        $this->viewData['title'] = "Liste d'état des lieux";
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());

        return view('data/etat/index', $this->viewData);
    }
    
    public function add()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(5);

        $this->viewData['url_list'] = "";
        $this->viewData['js'] = "Volontaire.js";
        $this->viewData['route'] = "/data/etat/save";
        $this->viewData['etat'] =  $this->initEtat($id);
        $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
      
        if ($id > 0)
            $title = "Editer un état de lieu";
        else
            $title = "Nouveau état de lieu";

        $this->viewData['title'] = $title;

        return view('data/etat/add', $this->viewData);
    }

    public function save()
    {
        $id = $this->request->getPost('id');

        if (!$this->validate($this->_rules())) {

            $this->viewData['url_list'] = "/data/etat/data";
            $this->viewData['js'] = "";
            $this->viewData['route'] = "/data/etat/save";
            $this->viewData['etat'] =  $this->initEtat($id);
            $this->viewData['branches'] = CommonTrait::formatData(new BrancheModel());
            
            if ($id > 0)
                $title = "Editer un état des lieux";
            else
                $title = "Nouveau état des lieux";

            $this->viewData['title'] = $title;

            return view('data/etat/add', $this->viewData);
        } else {

            $uneEtat = $this->uneEtat();
            if ($id) {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'état de lieux a été modifiée</div>");
                $stmt = $this->etatLieuModel->update($id, $uneEtat);
            } else {
                $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'état des lieux a été bien enregistrée</div>");
                $stmt = $this->etatLieuModel->save($uneEtat);
            }

            if ($stmt) {
                return redirect()->to(site_url($this->viewData['locale'] . '/data/etat'));
            }
        }
    }


    public function delete()
    {
        $uri = current_url(true);
        $id = $uri->getSegment(6);

        $stmnt = $this->etatLieuModel->delete($id);

        if ($stmnt) {
            $this->session->setFlashdata('message', "<div class='row alert alert-success'>L'état des lieux a été suprimée</div>");
            return redirect()->to(site_url($this->viewData['locale'] . '/data/etat'));
        }
    }

    private function uneEtat()
    {
        $etat = [];
        $etat['brancheId'] = $this->request->getPost('brancheId');
        $etat['communeId'] = $this->request->getPost('communeId');
        $etat['nbCEPV'] = $this->request->getPost('nbCEPV');
        $etat['nbReponsPSS'] = $this->request->getPost('nbReponsPSS');
        $etat['nbReponsPS'] = $this->request->getPost('nbReponsPS');
        $etat['nbReponsEPIC'] = $this->request->getPost('nbReponsEPIC');
        $etat['nbReponsWASH'] = $this->request->getPost('nbReponsWASH');
        $etat['nbReponsSante'] = $this->request->getPost('nbReponsSante');
        $etat['nbReponsConflit'] = $this->request->getPost('nbReponsConflit');
        $etat['nbUniteColline'] = $this->request->getPost('nbUniteColline');
        $etat['effectifAdultMascul'] = $this->request->getPost('effectifAdultMascul');
        $etat['effectifAdultFemin'] = $this->request->getPost('effectifAdultFemin');
        $etat['effectifJeuneMascul'] = $this->request->getPost('effectifJeuneMascul');
        $etat['effectifJeuneFemin'] = $this->request->getPost('effectifJeuneFemin');
        $etat['nbVolontHandicapH'] = $this->request->getPost('nbVolontHandicapH');
        $etat['nbVolontHandicapF'] = $this->request->getPost('nbVolontHandicapF');
        $etat['userId'] = \auth()->user()->id;
       
        $file = $this->request->getFile('pathFile');
        if ($file->isValid() && !$file->hasMoved()) {
            $newName = $file->getRandomName();
            $file->move('files/etat_lieux', $newName);
            $etat['pathFile'] = $newName;
        }

        return $etat;
    }

    private function initEtat($id = NULL)
    {
        $etat = NULL;
        if ($id) {
            $etat = $this->etatLieuModel->find($id);
        } else {
            $etat = new stdClass();
            $etat->id = 0;
            $etat->brancheId = 0;
            $etat->communeId = 0;
            $etat->nbCEPV = 0;
            $etat->nbReponsPSS = 0;
            $etat->nbReponsPS = 0;
            $etat->nbReponsEPIC = 0;
            $etat->nbReponsWASH = 0;
            $etat->nbReponsSante = 0;
            $etat->nbReponsConflit = 0;
            $etat->nbUniteColline = 0;
            $etat->effectifAdultMascul = 0;
            $etat->effectifAdultFemin = 0;
            $etat->effectifJeuneMascul = 0;
            $etat->effectifJeuneFemin = 0;
            $etat->nbVolontHandicapH = 0;
            $etat->nbVolontHandicapF = 0;
            $etat->pathFile = "";
        }

        return $etat;
    }

    private function _rules()
    {
        $rules = [];
        $rules['brancheId'] = ['rules'  => 'required'];
        $rules['communeId'] = ['rules'  => 'required'];
        $rules['nbCEPV'] = ['rules'  => 'required'];
        $rules['nbReponsPSS'] = ['rules'  => 'required'];
        $rules['nbReponsEPIC'] = ['rules'  => 'required'];
        $rules['nbReponsWASH'] = ['rules'  => 'required'];
        $rules['nbReponsSante'] = ['rules'  => 'required'];
        $rules['nbReponsConflit'] = ['rules'  => 'required'];
        $rules['nbUniteColline'] = ['rules'  => 'required'];
        $rules['effectifAdultMascul'] = ['rules'  => 'required'];
        $rules['effectifAdultFemin'] = ['rules'  => 'required'];
        $rules['effectifJeuneMascul'] = ['rules'  => 'required'];
        $rules['effectifJeuneFemin'] = ['rules'  => 'required'];
        $rules['nbVolontHandicapH'] = ['rules'  => 'required'];
        $rules['nbVolontHandicapF'] = ['rules'  => 'required'];

        return $rules;
    }

    function fecth()
    {
        $search_column = ['brancheId', 'communeId', 'nbCEPV', 'nbReponsPSS', 'nbReponsEPIC', 'nbReponsWASH', 'nbReponsSante', 'nbReponsConflit', 'nbUniteColline', 'effectifAdultMascul', 'effectifAdultFemin', 'effectifJeuneMascul', 'effectifJeuneFemin', 'nbVolontHandicapH', 'nbVolontHandicapF', 'pathFile'];
        $order_column = ['brancheId', 'communeId', 'nbCEPV', 'nbReponsPSS', 'nbReponsEPIC', 'nbReponsWASH', 'nbReponsSante', 'nbReponsConflit', 'nbUniteColline', 'effectifAdultMascul', 'effectifAdultFemin', 'effectifJeuneMascul', 'effectifJeuneFemin', 'nbVolontHandicapH', 'nbVolontHandicapF', 'pathFile'];

        $etatMdl = $this->etatLieuModel;
        if (\auth()->user()->inGroup('brancheManager')) {
            $etatMdl->where('brancheId', \auth()->user()->brancheId);
        }

        if (\auth()->user()->inGroup('communeManager')) {
            $etatMdl->where('communeId', \auth()->user()->communeId);
        }

        if ($this->request->getPost('brancheId') > 0) {
            $etatMdl->where('brancheId', $this->request->getPost('brancheId'));
        }

        $fetch_data = DatatableQuery::build_query($etatMdl, $search_column, $order_column);
        $output = $this->formatEtat($this->viewData['locale'], $fetch_data);

        return $this->response->setJSON($output);
    }

    private function formatEtat($locale, $datas)
    {
        $i = 1;
        $array_data = array();
        if (!empty($datas['data'])) {
            foreach ($datas['data'] as $data) {

                $sub_array = array();

                $sub_array[] =  $i;
                $sub_array[] =  CommonTrait::formatData(new BrancheModel(), $data->brancheId);
                $sub_array[] =  CommonTrait::formatData(new CommuneModel(), $data->communeId);
                $sub_array[] =  $data->nbCEPV;
                $sub_array[] =  $data->nbReponsPSS;
                $sub_array[] =  $data->nbReponsPS;
                $sub_array[] =  $data->nbReponsEPIC;
                $sub_array[] =  $data->nbReponsWASH;
                $sub_array[] =  $data->nbReponsSante;
                $sub_array[] =  $data->nbReponsConflit;
                $sub_array[] = !empty($data->pathFile)? anchor(base_url()."files/etat_lieux/".$data->pathFile, '...'.substr($data->pathFile,-8), "target='_blank'"):"";
    
                $option = "-";
                if (\auth()->user()->inGroup('hr', 'admin') || (\auth()->user()->hasPermission('params.edit') && \auth()->user()->hasPermission('params.delete'))) {
                    $option = anchor($locale . "/data/etat/add/" . $data->id, "<i class='fa fa-edit'></i>", ['class' => 'text-info']) . " "
                        . anchor($locale . "/data/etat/delete/" . $data->id, "<i class='fa fa-trash'></i>", ['class' => 'text-danger']);
                }
                $sub_array[] = $option;

                $array_data[] = $sub_array;
                $i++;
            }
        }

        return [
            "draw" => intval($_POST["draw"]),
            "recordsTotal" => $datas['recordsTotal'],
            "recordsFiltered" => $datas['recordsFiltered'],
            "data" => $array_data
        ];
    }
    function export(){
        $uri = current_url(true);
        $brancheId = $uri->getSegment(5);

        if ($brancheId > 0) {
            $branche = CommonTrait::formatData(new BrancheModel(), $brancheId);
        } else {
            $branche = "Toutes les branches";
        }

        $fileName = str_replace(' ', '', $branche);        
        $csv_file_name = "rapportExport/etat_lieu/etat_lieu_" . $fileName . ".csv";
        $file = fopen($csv_file_name, 'w');

        $arrays = [];
        $arrays[0] = ['', $this->formatText("Etat des lieux")];
        $arrays[1] = ['', $this->formatText($branche)];
        $arrays[2] = [];
        $arrays[3] = [
            '#',   
            $this->formatText('Branche'),  
            $this->formatText('Commune'), 
            $this->formatText('Nombre de CEPV'), 
            $this->formatText('Nombre équipe (s) de réponse PSS'), 
            $this->formatText('Nombre équipe (s) de réponse  PS'), 
            $this->formatText('Nombre équipe (s)  de réponse EPIC'), 
            $this->formatText("Nombre équipe (s) de réponse Wash"), 
            $this->formatText('Nombre équipe (s) de réponse Santé'),  
            $this->formatText('Nombre équipe (s) de réponse gestion pacifique des conflits'), 
            $this->formatText('Nombre Unités Collinaires'), 
            $this->formatText("Effectif Adultes Masculins"), 
            $this->formatText('Effectif Adultes Féminins'),
            $this->formatText('Effectif Jeunes Masculins'),
            $this->formatText('Effectif Jeunes Fémins'),
            $this->formatText('Nombre de volontaires vivant(s) avec handicaps (H)'),
            $this->formatText('Nombre de volontaires vivant(s) avec handicaps (F)')
        ];

        if ($brancheId > 0) {
            $etatLieux = $this->etatLieuModel->where('brancheId', $brancheId)->findAll();
        } else {
            $etatLieux = $this->etatLieuModel->findAll();
        }

        $i = 1;
        foreach ($etatLieux as $etatLieu) {
            $sub_array = [];
            $sub_array[] = $i;
            $sub_array[] =  $this->formatText(CommonTrait::formatData(new BrancheModel(), $etatLieu->brancheId));
            $sub_array[] =  $this->formatText(CommonTrait::formatData(new CommuneModel(), $etatLieu->communeId));
            $sub_array[] =  $etatLieu->nbCEPV;
            $sub_array[] =  $etatLieu->nbReponsPSS;
            $sub_array[] =  $etatLieu->nbReponsPS;
            $sub_array[] =  $etatLieu->nbReponsEPIC;
            $sub_array[] =  $etatLieu->nbReponsWASH;
            $sub_array[] =  $etatLieu->nbReponsSante;
            $sub_array[] =  $etatLieu->nbReponsConflit;
            $sub_array[] =  $etatLieu->nbUniteColline;
            $sub_array[] =  $etatLieu->effectifAdultMascul;
            $sub_array[] =  $etatLieu->effectifAdultFemin;
            $sub_array[] =  $etatLieu->effectifJeuneMascul;
            $sub_array[] =  $etatLieu->effectifJeuneFemin;
            $sub_array[] =  $etatLieu->nbVolontHandicapH;
            $sub_array[] =  $etatLieu->nbVolontHandicapF;

            $arrays[] = $sub_array;
            $i++;
        }


        foreach ($arrays as $array) {
            fputcsv($file, $array);
        }
        fclose($file);
        $btn = anchor($csv_file_name, "Exporter un <i class='fa fa-file-csv'></i>", ['class' => 'text-success', 'target' => '_blank']);

        $status = 0;
        if (fileExists($csv_file_name)) {
            $status = 1;
        }

        return $this->response->setJSON(['status' => $status, 'btnExport' => $btn]);
    }

    function formatText($string = "")
    {
        return UConverter::transcode($string, 'ISO-8859-1', 'UTF8');
    }
}
